// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Collections.Generic;

using Internal.TypeSystem;

namespace Internal.IL.Stubs
{
    /// <summary>
    /// Synthetic method override of "int ValueType.__GetFieldHelper(Int32, out EETypePtr)". This method is injected
    /// into all value types that cannot have their Equals(object) and GetHashCode() methods operate on individual
    /// bytes. The purpose of the override is to provide access to the value types' fields and their types.
    /// </summary>
    public sealed partial class ValueTypeGetFieldHelperMethodOverride : ILStubMethod
    {
        private DefType _owningType;
        private MethodSignature _signature;

        internal ValueTypeGetFieldHelperMethodOverride(DefType owningType)
        {
            _owningType = owningType;
        }

        public override TypeSystemContext Context
        {
            get
            {
                return _owningType.Context;
            }
        }

        public override TypeDesc OwningType
        {
            get
            {
                return _owningType;
            }
        }

        public override MethodSignature Signature
        {
            get
            {
                if (_signature == null)
                {
                    TypeSystemContext context = _owningType.Context;
                    TypeDesc int32Type = context.GetWellKnownType(WellKnownType.Int32);
                    TypeDesc eeTypePtrType = context.SystemModule.GetKnownType("System", "EETypePtr");

                    _signature = new MethodSignature(0, 0, int32Type, new[] {
                        int32Type,
                        eeTypePtrType.MakeByRefType()
                    });
                }

                return _signature;
            }
        }

        public override MethodIL EmitIL()
        {
            TypeDesc owningType = _owningType.InstantiateAsOpen();

            ILEmitter emitter = new ILEmitter();

            TypeDesc eeTypePtrType = Context.SystemModule.GetKnownType("System", "EETypePtr");
            MethodDesc eeTypePtrOfMethod = eeTypePtrType.GetKnownMethod("EETypePtrOf", null);
            ILToken eeTypePtrToken = emitter.NewToken(eeTypePtrType);

            var switchStream = emitter.NewCodeStream();
            var getFieldStream = emitter.NewCodeStream();

            ArrayBuilder<ILCodeLabel> fieldGetters = new ArrayBuilder<ILCodeLabel>();
            foreach (FieldDesc field in owningType.GetFields())
            {
                if (field.IsStatic)
                    continue;

                ILCodeLabel label = emitter.NewCodeLabel();
                fieldGetters.Add(label);

                getFieldStream.EmitLabel(label);
                getFieldStream.EmitLdArg(2);

                // We need something we can instantiate EETypePtrOf over. Also, the classlib
                // code doesn't handle pointers.
                TypeDesc boxableFieldType = field.FieldType;
                if (boxableFieldType.IsPointer || boxableFieldType.IsFunctionPointer)
                    boxableFieldType = Context.GetWellKnownType(WellKnownType.IntPtr);

                MethodDesc ptrOfField = eeTypePtrOfMethod.MakeInstantiatedMethod(boxableFieldType);
                getFieldStream.Emit(ILOpcode.call, emitter.NewToken(ptrOfField));

                getFieldStream.Emit(ILOpcode.stobj, eeTypePtrToken);

                getFieldStream.EmitLdArg(0);
                getFieldStream.Emit(ILOpcode.ldflda, emitter.NewToken(field));

                getFieldStream.EmitLdArg(0);

                getFieldStream.Emit(ILOpcode.sub);

                getFieldStream.Emit(ILOpcode.ret);
            }

            if (fieldGetters.Count > 0)
            {
                switchStream.EmitLdArg(1);
                switchStream.EmitSwitch(fieldGetters.ToArray());
            }

            switchStream.EmitLdc(fieldGetters.Count);
            
            switchStream.Emit(ILOpcode.ret);

            return emitter.Link(this);
        }

        public override Instantiation Instantiation
        {
            get
            {
                return Instantiation.Empty;
            }
        }

        public override bool IsVirtual
        {
            get
            {
                return true;
            }
        }

        public override string Name
        {
            get
            {
                return "__GetFieldHelper";
            }
        }
    }
}
